<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;

class BrandingController extends Controller
{
    private const FRONTEND_DEFAULTS = [
        'frontend_hero_title' => 'Science Laboratory',
        'frontend_hero_subtitle' => 'Smart workflow for patient records, orders, billing, and secure result access.',
        'frontend_contact_title' => 'Contact Us',
        'frontend_contact_description' => 'Reach our support team for lab, result, billing, or account help.',
        'frontend_contact_email' => '',
        'frontend_contact_phone' => '',
        'frontend_contact_address' => '',
        'frontend_whatsapp_number' => '',
        'frontend_facebook_url' => '',
        'frontend_instagram_url' => '',
        'frontend_x_url' => '',
        'frontend_linkedin_url' => '',
    ];

    public function edit(): View
    {
        $appName = Setting::getGlobal('app_name', config('app.name', 'Lab Result Portal'));
        $appLogoPath = Setting::getGlobal('app_logo_path');
        $frontend = [];

        foreach (self::FRONTEND_DEFAULTS as $key => $default) {
            $frontend[$key] = Setting::getGlobal($key, $default) ?? $default;
        }

        return view('admin.branding.edit', compact('appName', 'appLogoPath', 'frontend'));
    }

    public function update(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'app_name' => ['required', 'string', 'max:255'],
            'logo' => ['nullable', 'image', 'max:4096'],
            'frontend_hero_title' => ['required', 'string', 'max:255'],
            'frontend_hero_subtitle' => ['required', 'string', 'max:600'],
            'frontend_contact_title' => ['required', 'string', 'max:255'],
            'frontend_contact_description' => ['nullable', 'string', 'max:600'],
            'frontend_contact_email' => ['nullable', 'email', 'max:255'],
            'frontend_contact_phone' => ['nullable', 'string', 'max:50'],
            'frontend_contact_address' => ['nullable', 'string', 'max:255'],
            'frontend_whatsapp_number' => ['nullable', 'string', 'max:30'],
            'frontend_facebook_url' => ['nullable', 'url', 'max:255'],
            'frontend_instagram_url' => ['nullable', 'url', 'max:255'],
            'frontend_x_url' => ['nullable', 'url', 'max:255'],
            'frontend_linkedin_url' => ['nullable', 'url', 'max:255'],
        ]);

        Setting::setGlobal('app_name', $data['app_name']);

        if ($request->hasFile('logo')) {
            $oldLogoPath = Setting::getGlobal('app_logo_path');
            if ($oldLogoPath) {
                Storage::disk('public')->delete($oldLogoPath);
            }

            $newLogoPath = $request->file('logo')->store('branding', 'public');
            Setting::setGlobal('app_logo_path', $newLogoPath);
        }

        foreach (array_keys(self::FRONTEND_DEFAULTS) as $key) {
            Setting::setGlobal($key, $data[$key] ?? '');
        }

        return redirect()->route('admin.branding.edit')->with('success', 'Branding and frontend settings updated successfully.');
    }
}
