<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Lab;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class UserController extends Controller
{
    public function index(Request $request): View
    {
        $query = User::query()->with('lab:id,name,branch_name')->latest('id');

        if ($term = trim((string) $request->query('q'))) {
            $query->where(function ($q) use ($term): void {
                $q->where('name', 'like', "%{$term}%")
                    ->orWhere('email', 'like', "%{$term}%")
                    ->orWhere('phone', 'like', "%{$term}%")
                    ->orWhere('role', 'like', "%{$term}%");
            });
        }

        $users = $query->paginate(15)->withQueryString();
        $labs = Lab::query()->orderBy('name')->get(['id', 'name', 'branch_name']);

        return view('admin.users.index', compact('users', 'labs'));
    }

    public function create(): View
    {
        $labs = Lab::query()->orderBy('name')->get(['id', 'name', 'branch_name']);
        return view('admin.users.create', compact('labs'));
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $this->validateUser($request);

        $user = User::query()->create([
            'name' => $data['name'],
            'email' => $data['email'],
            'phone' => $data['phone'] ?? null,
            'role' => $data['role'],
            'lab_id' => $this->resolveLabId($data),
            'is_active' => (bool) ($data['is_active'] ?? true),
            'email_verified_at' => now(),
            'password' => Hash::make($data['password']),
        ]);

        return redirect()->route('admin.users.index')->with('success', "User {$user->name} created.");
    }

    public function edit(User $user): View
    {
        $labs = Lab::query()->orderBy('name')->get(['id', 'name', 'branch_name']);
        return view('admin.users.edit', compact('user', 'labs'));
    }

    public function update(Request $request, User $user): RedirectResponse
    {
        $data = $this->validateUser($request, $user->id, false);

        $payload = [
            'name' => $data['name'],
            'email' => $data['email'],
            'phone' => $data['phone'] ?? null,
            'role' => $data['role'],
            'lab_id' => $this->resolveLabId($data),
            'is_active' => (bool) ($data['is_active'] ?? false),
        ];

        if (! empty($data['password'])) {
            $payload['password'] = Hash::make($data['password']);
        }

        $user->update($payload);

        return redirect()->route('admin.users.index')->with('success', "User {$user->name} updated.");
    }

    public function destroy(Request $request, User $user): RedirectResponse
    {
        if ((int) $request->user()->id === (int) $user->id) {
            return back()->withErrors(['user' => 'You cannot delete your own account.']);
        }

        $user->delete();

        return redirect()->route('admin.users.index')->with('success', 'User deleted.');
    }

    public function switchRole(Request $request, User $user): RedirectResponse
    {
        $data = $request->validate([
            'role' => ['required', Rule::in(['super_admin', 'lab_manager', 'lab_staff'])],
            'lab_id' => ['nullable', 'exists:labs,id'],
        ]);

        if (in_array($data['role'], ['lab_manager', 'lab_staff'], true) && empty($data['lab_id'])) {
            return back()->withErrors(['role' => 'Lab assignment is required for lab manager/staff.']);
        }

        $user->update([
            'role' => $data['role'],
            'lab_id' => $data['role'] === 'super_admin' ? null : (int) $data['lab_id'],
        ]);

        return back()->with('success', "Role updated for {$user->name}.");
    }

    /**
     * @return array<string, mixed>
     */
    private function validateUser(Request $request, ?int $userId = null, bool $requirePassword = true): array
    {
        $passwordRules = $requirePassword
            ? ['required', 'string', 'min:8', 'confirmed']
            : ['nullable', 'string', 'min:8', 'confirmed'];

        return $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users', 'email')->ignore($userId)],
            'phone' => ['nullable', 'string', 'max:30', Rule::unique('users', 'phone')->ignore($userId)],
            'role' => ['required', Rule::in(['super_admin', 'lab_manager', 'lab_staff'])],
            'lab_id' => [
                Rule::requiredIf(fn () => in_array((string) $request->input('role'), ['lab_manager', 'lab_staff'], true)),
                'nullable',
                'exists:labs,id',
            ],
            'is_active' => ['nullable', 'boolean'],
            'password' => $passwordRules,
        ]);
    }

    /**
     * @param  array<string, mixed>  $data
     */
    private function resolveLabId(array $data): ?int
    {
        if ($data['role'] === 'super_admin') {
            return null;
        }

        return ! empty($data['lab_id']) ? (int) $data['lab_id'] : null;
    }
}
