<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\NotificationLog;
use App\Models\Otp;
use App\Models\Patient;
use App\Models\Order;
use App\Models\LabResult;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Response;
use Illuminate\View\View;

class PatientOtpLoginController extends Controller
{
    private const OTP_EXPIRY_MINUTES = 10;
    private const OTP_MAX_ATTEMPTS = 5;
    private const OTP_RESEND_COOLDOWN_SECONDS = 60;
    private const OTP_SESSION_KEY = 'patient_otp_id';

    public function showRequestForm(): View|RedirectResponse
    {
        if (Auth::guard('patient')->check()) {
            return redirect()->route('patient.portal');
        }

        return view('patient.auth.request-otp');
    }

    public function requestOtp(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'identifier' => ['required', 'string', 'max:255'],
        ]);

        $identifier = trim((string) $validated['identifier']);
        $patient = Patient::query()
            ->where('email', $identifier)
            ->orWhere('phone', $identifier)
            ->first();

        if (! $patient) {
            return back()
                ->withErrors(['identifier' => 'No patient account found for the provided email or phone.'])
                ->withInput();
        }

        $latestOtp = Otp::query()
            ->where('identifier', $this->otpIdentifier($patient->id))
            ->whereNull('consumed_at')
            ->latest('id')
            ->first();

        if ($latestOtp && $latestOtp->created_at->gt(now()->subSeconds(self::OTP_RESEND_COOLDOWN_SECONDS))) {
            return back()
                ->withErrors(['identifier' => 'Please wait a minute before requesting another OTP.'])
                ->withInput();
        }

        $channels = $this->resolveChannels($patient);

        if ($channels === []) {
            return back()
                ->withErrors(['identifier' => 'Patient account has no email or phone available for OTP delivery.'])
                ->withInput();
        }

        $code = (string) random_int(100000, 999999);

        $otp = Otp::query()->create([
            'patient_id' => $patient->id,
            'identifier' => $this->otpIdentifier($patient->id),
            'channel' => implode(',', $channels),
            'code_hash' => Hash::make($code),
            'plain_code' => $code,
            'attempts' => 0,
            'expires_at' => now()->addMinutes(self::OTP_EXPIRY_MINUTES),
            'locked_until' => null,
            'consumed_at' => null,
        ]);

        $message = "Your Lab Result Portal OTP is {$code}. It expires in ".self::OTP_EXPIRY_MINUTES.' minutes.';
        $this->dispatchOtp($patient, $channels, $message);

        $request->session()->put(self::OTP_SESSION_KEY, $otp->id);

        return redirect()
            ->route('patient.otp.verify')
            ->with('status', 'OTP sent via available channels (email/SMS/WhatsApp).');
    }

    public function showVerifyForm(Request $request): View|RedirectResponse
    {
        if (Auth::guard('patient')->check()) {
            return redirect()->route('patient.portal');
        }

        if (! $request->session()->has(self::OTP_SESSION_KEY)) {
            return redirect()->route('patient.login');
        }

        return view('patient.auth.verify-otp');
    }

    public function verifyOtp(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'otp' => ['required', 'digits:6'],
        ]);

        $otpId = (int) $request->session()->get(self::OTP_SESSION_KEY);
        $otp = Otp::query()->find($otpId);

        if (! $otp || $otp->consumed_at !== null) {
            return redirect()
                ->route('patient.login')
                ->withErrors(['otp' => 'OTP session is no longer valid. Request a new code.']);
        }

        if ($otp->locked_until && $otp->locked_until->isFuture()) {
            return back()->withErrors([
                'otp' => 'Too many attempts. Try again at '.$otp->locked_until->format('H:i:s'),
            ]);
        }

        if ($otp->expires_at->isPast()) {
            return redirect()
                ->route('patient.login')
                ->withErrors(['otp' => 'OTP has expired. Request a new code.']);
        }

        if (! Hash::check((string) $validated['otp'], $otp->code_hash)) {
            $attempts = $otp->attempts + 1;
            $otp->attempts = $attempts;

            if ($attempts >= self::OTP_MAX_ATTEMPTS) {
                $otp->locked_until = now()->addMinutes(self::OTP_EXPIRY_MINUTES);
            }

            $otp->save();

            return back()->withErrors(['otp' => 'Invalid OTP.']);
        }

        $patientId = $this->parsePatientId($otp->identifier);
        $patient = $patientId ? Patient::query()->find($patientId) : null;

        if (! $patient) {
            return redirect()
                ->route('patient.login')
                ->withErrors(['otp' => 'Unable to resolve patient account.']);
        }

        $otp->forceFill([
            'consumed_at' => now(),
            'locked_until' => null,
        ])->save();

        Auth::guard('patient')->login($patient);
        $request->session()->regenerate();
        $request->session()->forget(self::OTP_SESSION_KEY);

        return redirect()->route('patient.portal');
    }

    public function portal(Request $request): View
    {
        /** @var Patient $patient */
        $patient = Auth::guard('patient')->user();

        $orders = $patient->orders()
            ->with(['lab:id,name,branch_name', 'results:id,order_id,version,file_path,ready_at,created_at'])
            ->latest('id')
            ->get();

        return view('patient.portal', compact('patient', 'orders'));
    }

    public function logout(Request $request): RedirectResponse
    {
        Auth::guard('patient')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('patient.login');
    }

    public function downloadResult(Order $order, LabResult $result)
    {
        /** @var Patient $patient */
        $patient = Auth::guard('patient')->user();

        if ((int) $order->patient_id !== (int) $patient->id || (int) $result->order_id !== (int) $order->id) {
            abort(403);
        }

        $path = storage_path('app/public/'.$result->file_path);
        if (! file_exists($path)) {
            abort(404, 'Result file not found.');
        }

        return Response::download($path, basename($path));
    }

    public function downloadTransactionsReceipt()
    {
        /** @var Patient $patient */
        $patient = Auth::guard('patient')->user();

        $orders = $patient->orders()
            ->with(['lab:id,name,branch_name', 'payments:id,order_id,amount,method,reference,paid_at,created_at'])
            ->latest('id')
            ->get();

        $payments = $orders->flatMap(function ($order) {
            return $order->payments->map(function ($payment) use ($order) {
                return [
                    'order_code' => $order->order_code,
                    'lab' => trim((string) optional($order->lab)->name.' '.(string) optional($order->lab)->branch_name),
                    'amount' => (float) $payment->amount,
                    'method' => (string) $payment->method,
                    'reference' => (string) ($payment->reference ?? ''),
                    'paid_at' => $payment->paid_at ?? $payment->created_at,
                ];
            });
        })->sortByDesc(fn (array $row) => optional($row['paid_at'])->timestamp ?? 0)->values();

        $totalPaid = (float) $payments->sum('amount');
        $totalBilled = (float) $orders->sum('total');
        $totalBalance = max($totalBilled - $totalPaid, 0);

        $pdf = Pdf::loadView('patient.transactions-receipt', [
            'patient' => $patient,
            'orders' => $orders,
            'payments' => $payments,
            'totalPaid' => $totalPaid,
            'totalBilled' => $totalBilled,
            'totalBalance' => $totalBalance,
        ]);

        return $pdf->download('transactions_receipt_'.$patient->id.'_'.now()->format('Ymd_His').'.pdf');
    }

    /**
     * @return array<int, string>
     */
    private function resolveChannels(Patient $patient): array
    {
        $channels = [];

        if ($patient->email) {
            $channels[] = 'email';
        }

        if ($patient->phone) {
            $channels[] = 'sms';
            $channels[] = 'whatsapp';
        }

        return $channels;
    }

    /**
     * @param  array<int, string>  $channels
     */
    private function dispatchOtp(Patient $patient, array $channels, string $message): void
    {
        foreach ($channels as $channel) {
            try {
                if ($channel === 'email' && $patient->email) {
                    Mail::raw($message, static function ($mail) use ($patient): void {
                        $mail->to($patient->email)->subject('Your Login OTP');
                    });

                    $this->logNotification($channel, 'sent', $message);
                    continue;
                }

                if ($channel === 'sms' && $patient->phone) {
                    $this->dispatchHttpChannel(
                        channel: 'sms',
                        endpoint: (string) config('services.patient_sms.endpoint'),
                        payload: ['to' => $patient->phone, 'message' => $message]
                    );
                    continue;
                }

                if ($channel === 'whatsapp' && $patient->phone) {
                    $this->dispatchHttpChannel(
                        channel: 'whatsapp',
                        endpoint: (string) config('services.patient_whatsapp.endpoint'),
                        payload: ['to' => $patient->phone, 'message' => $message]
                    );
                }
            } catch (\Throwable $e) {
                Log::warning('OTP dispatch failed', ['channel' => $channel, 'error' => $e->getMessage()]);
                $this->logNotification($channel, 'failed', $message, $e->getMessage());
            }
        }
    }

    /**
     * @param  array<string, string>  $payload
     */
    private function dispatchHttpChannel(string $channel, string $endpoint, array $payload): void
    {
        if ($endpoint === '') {
            // No provider endpoint configured. Keep a simulated log entry in non-integrated environments.
            $this->logNotification($channel, 'simulated', $payload['message'] ?? null, 'Provider endpoint not configured.');
            return;
        }

        $response = Http::asJson()->post($endpoint, $payload);
        $status = $response->successful() ? 'sent' : 'failed';

        $this->logNotification(
            $channel,
            $status,
            $payload['message'] ?? null,
            json_encode(['status' => $response->status(), 'body' => $response->body()], JSON_THROW_ON_ERROR)
        );
    }

    private function logNotification(string $channel, string $status, ?string $message = null, ?string $providerResponse = null): void
    {
        NotificationLog::query()->create([
            'channel' => $channel,
            'status' => $status,
            'message' => $message,
            'provider_response' => $providerResponse,
            'sent_at' => Carbon::now(),
        ]);
    }

    private function otpIdentifier(int $patientId): string
    {
        return 'patient:'.$patientId;
    }

    private function parsePatientId(string $identifier): ?int
    {
        if (! str_starts_with($identifier, 'patient:')) {
            return null;
        }

        $id = (int) substr($identifier, strlen('patient:'));

        return $id > 0 ? $id : null;
    }
}
