<?php

namespace App\Http\Controllers;

use App\Models\Lab;
use App\Models\Order;
use App\Models\Patient;
use App\Models\Payment;
use App\Models\User;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Carbon;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function __invoke(Request $request): View
    {
        $user = $request->user();
        $role = $user->role;
        $labId = $user->lab_id;

        $ordersQuery = Order::query();
        $patientsQuery = Patient::query();
        $paymentsQuery = Payment::query()->join('orders', 'orders.id', '=', 'payments.order_id');

        if (in_array($role, ['lab_manager', 'lab_staff'], true)) {
            $ordersQuery->where('lab_id', $labId);
            $patientsQuery->where('lab_id', $labId);
            $paymentsQuery->where('orders.lab_id', $labId);
        }

        $todayStart = Carbon::today();
        $todayEnd = Carbon::today()->endOfDay();
        $monthStart = Carbon::now()->startOfMonth();
        $monthEnd = Carbon::now()->endOfMonth();

        $stats = [
            'labs' => $role === 'super_admin' ? Lab::query()->count() : null,
            'staff' => $role === 'super_admin'
                ? User::query()->whereIn('role', ['lab_manager', 'lab_staff'])->count()
                : User::query()->where('lab_id', $labId)->count(),
            'patients' => (clone $patientsQuery)->count(),
            'orders_total' => (clone $ordersQuery)->count(),
            'orders_ready' => (clone $ordersQuery)->where('status', 'ready')->count(),
            'orders_pending' => (clone $ordersQuery)->whereIn('status', ['draft', 'pending'])->count(),
            'revenue' => (float) (clone $paymentsQuery)->sum('payments.amount'),
            'revenue_daily' => (float) (clone $paymentsQuery)
                ->whereBetween('payments.paid_at', [$todayStart, $todayEnd])
                ->sum('payments.amount'),
            'revenue_monthly' => (float) (clone $paymentsQuery)
                ->whereBetween('payments.paid_at', [$monthStart, $monthEnd])
                ->sum('payments.amount'),
        ];

        $recentOrders = (clone $ordersQuery)
            ->with(['patient:id,full_name', 'lab:id,name,branch_name'])
            ->latest('id')
            ->take(8)
            ->get();

        return view('dashboard', compact('stats', 'recentOrders', 'role'));
    }
}
