<?php

namespace App\Http\Controllers\Frontdesk;

use App\Http\Controllers\Controller;
use App\Models\Lab;
use App\Models\LabTestPrice;
use App\Models\Order;
use App\Models\Patient;
use App\Models\TestCatalog;
use App\Services\OrderTotalsService;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Symfony\Component\HttpFoundation\StreamedResponse;

class OrderController extends Controller
{
    public function __construct(private readonly OrderTotalsService $totalsService)
    {
    }

    public function index(Request $request): View
    {
        $orders = $this->queryOrders($request)
            ->with(['patient:id,full_name,phone,email', 'lab:id,name,branch_name'])
            ->latest('id')
            ->paginate(15)
            ->withQueryString();

        return view('frontdesk.orders.index', compact('orders'));
    }

    public function create(Request $request, Patient $patient): View
    {
        $lab = $this->resolveLab($request);
        $this->ensurePatientMatchesLab($patient, $lab->id);
        $tests = TestCatalog::query()->where('is_active', true)->orderBy('name')->get();
        $overrides = LabTestPrice::query()
            ->where('lab_id', $lab->id)
            ->get()
            ->keyBy('test_id');

        return view('frontdesk.orders.create', compact('patient', 'tests', 'overrides', 'lab'));
    }

    public function store(Request $request, Patient $patient): RedirectResponse
    {
        $lab = $this->resolveLab($request);
        $this->ensurePatientMatchesLab($patient, $lab->id);

        $validated = $request->validate([
            'discount' => ['nullable', 'numeric', 'min:0'],
            'tests' => ['required', 'array', 'min:1'],
            'tests.*.test_id' => ['required', 'exists:test_catalog,id'],
            'tests.*.qty' => ['required', 'integer', 'min:1'],
        ]);

        $order = Order::query()->create([
            'lab_id' => $lab->id,
            'patient_id' => $patient->id,
            'order_code' => $this->nextOrderCode($lab->id),
            'status' => 'pending',
            'discount' => (float) ($validated['discount'] ?? 0),
            'created_by' => $request->user()->id,
        ]);

        foreach ($validated['tests'] as $row) {
            $test = TestCatalog::query()->findOrFail((int) $row['test_id']);
            $qty = (int) $row['qty'];
            $price = $this->resolveTestPrice($lab->id, $test);

            $order->items()->create([
                'test_id' => $test->id,
                'qty' => $qty,
                'price' => $price,
                'line_total' => $price * $qty,
            ]);
        }

        $this->totalsService->recalculate($order);

        return redirect()->route('frontdesk.orders.show', $order)->with('success', 'Order created.');
    }

    public function show(Request $request, Order $order): View
    {
        $this->ensureOrderAccessible($request, $order);

        $order->load([
            'patient',
            'lab',
            'items.test',
            'payments',
            'results',
        ]);

        $tests = TestCatalog::query()->where('is_active', true)->orderBy('name')->get();
        $overrides = LabTestPrice::query()->where('lab_id', $order->lab_id)->get()->keyBy('test_id');

        return view('frontdesk.orders.show', compact('order', 'tests', 'overrides'));
    }

    public function update(Request $request, Order $order): RedirectResponse
    {
        $this->ensureOrderAccessible($request, $order);

        $data = $request->validate([
            'status' => ['required', 'in:draft,pending,paid,ready,delivered,cancelled'],
            'discount' => ['nullable', 'numeric', 'min:0'],
        ]);

        $order->update([
            'status' => $data['status'],
            'discount' => (float) ($data['discount'] ?? 0),
            'ready_at' => $data['status'] === 'ready' ? now() : $order->ready_at,
            'delivered_at' => $data['status'] === 'delivered' ? now() : $order->delivered_at,
        ]);

        $this->totalsService->recalculate($order);

        return back()->with('success', 'Order updated.');
    }

    public function destroy(Request $request, Order $order): RedirectResponse
    {
        $this->ensureOrderAccessible($request, $order);

        foreach ($order->results as $result) {
            if ($result->file_path) {
                @unlink(storage_path('app/public/'.$result->file_path));
            }
        }

        $order->delete();

        return redirect()->route('frontdesk.orders.index')->with('success', 'Order deleted.');
    }

    public function exportCsv(Request $request): StreamedResponse
    {
        $rows = $this->queryOrders($request)
            ->with(['patient:id,full_name', 'lab:id,name,branch_name'])
            ->latest('id')
            ->get();

        return response()->streamDownload(static function () use ($rows): void {
            $handle = fopen('php://output', 'wb');
            fputcsv($handle, ['Code', 'Lab', 'Patient', 'Status', 'Total', 'Paid', 'Balance', 'Created']);

            foreach ($rows as $order) {
                fputcsv($handle, [
                    $order->order_code,
                    trim((string) optional($order->lab)->name.' '.(string) optional($order->lab)->branch_name),
                    optional($order->patient)->full_name,
                    $order->status,
                    $order->total,
                    $order->paid,
                    $order->balance,
                    optional($order->created_at)->format('Y-m-d H:i'),
                ]);
            }
            fclose($handle);
        }, 'orders_'.now()->format('Ymd_His').'.csv', ['Content-Type' => 'text/csv']);
    }

    public function exportPdf(Request $request)
    {
        $orders = $this->queryOrders($request)
            ->with(['patient:id,full_name', 'lab:id,name,branch_name'])
            ->latest('id')
            ->get();

        $pdf = Pdf::loadView('frontdesk.orders.export-pdf', compact('orders'));
        return $pdf->download('orders_'.now()->format('Ymd_His').'.pdf');
    }

    public function print(Request $request): View
    {
        $orders = $this->queryOrders($request)
            ->with(['patient:id,full_name', 'lab:id,name,branch_name'])
            ->latest('id')
            ->get();

        return view('frontdesk.orders.print', compact('orders'));
    }

    private function resolveLab(Request $request): Lab
    {
        $user = $request->user();

        if (in_array($user->role, ['lab_manager', 'lab_staff'], true)) {
            return Lab::query()->findOrFail((int) $user->lab_id);
        }

        $labId = (int) $request->input('lab_id');

        if ($labId > 0) {
            return Lab::query()->findOrFail($labId);
        }

        return Lab::query()->firstOrFail();
    }

    private function queryOrders(Request $request)
    {
        $query = Order::query();
        $user = $request->user();

        if (in_array($user->role, ['lab_manager', 'lab_staff'], true)) {
            $query->where('lab_id', $user->lab_id);
        } elseif ($labId = (int) $request->query('lab_id')) {
            $query->where('lab_id', $labId);
        }

        if ($status = trim((string) $request->query('status'))) {
            $query->where('status', $status);
        }

        if ($term = trim((string) $request->query('q'))) {
            $query->where(function ($q) use ($term): void {
                $q->where('order_code', 'like', "%{$term}%")
                    ->orWhereHas('patient', static fn ($p) => $p->where('full_name', 'like', "%{$term}%"));
            });
        }

        return $query;
    }

    private function ensureOrderAccessible(Request $request, Order $order): void
    {
        $user = $request->user();
        if (in_array($user->role, ['lab_manager', 'lab_staff'], true) && (int) $order->lab_id !== (int) $user->lab_id) {
            abort(403, 'Unauthorized order access.');
        }
    }

    private function resolveTestPrice(int $labId, TestCatalog $test): float
    {
        $override = LabTestPrice::query()
            ->where('lab_id', $labId)
            ->where('test_id', $test->id)
            ->value('price');

        return (float) ($override ?? $test->default_price);
    }

    private function nextOrderCode(int $labId): string
    {
        $prefix = 'LAB'.$labId;
        $random = strtoupper(Str::random(6));

        return $prefix.'-'.now()->format('YmdHis').'-'.$random;
    }

    private function ensurePatientMatchesLab(Patient $patient, int $labId): void
    {
        if ((int) $patient->lab_id !== $labId) {
            abort(403, 'Patient does not belong to the selected lab.');
        }
    }
}
