<?php

namespace App\Http\Controllers\Frontdesk;

use App\Http\Controllers\Controller;
use App\Models\LabTestPrice;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\TestCatalog;
use App\Services\OrderTotalsService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;

class OrderItemController extends Controller
{
    public function __construct(private readonly OrderTotalsService $totalsService)
    {
    }

    public function store(Request $request, Order $order): RedirectResponse
    {
        $this->ensureOrderAccessible($request, $order);

        $data = $request->validate([
            'test_id' => ['required', 'exists:test_catalog,id'],
            'qty' => ['required', 'integer', 'min:1'],
        ]);

        $test = TestCatalog::query()->findOrFail((int) $data['test_id']);
        $qty = (int) $data['qty'];
        $price = $this->resolveTestPrice((int) $order->lab_id, $test);

        $order->items()->create([
            'test_id' => $test->id,
            'qty' => $qty,
            'price' => $price,
            'line_total' => $qty * $price,
        ]);

        $this->totalsService->recalculate($order);

        return back()->with('success', 'Test item added.');
    }

    public function destroy(Request $request, Order $order, OrderItem $item): RedirectResponse
    {
        $this->ensureOrderAccessible($request, $order);

        if ((int) $item->order_id !== (int) $order->id) {
            abort(404);
        }

        $item->delete();
        $this->totalsService->recalculate($order);

        return back()->with('success', 'Test item removed.');
    }

    private function resolveTestPrice(int $labId, TestCatalog $test): float
    {
        $override = LabTestPrice::query()
            ->where('lab_id', $labId)
            ->where('test_id', $test->id)
            ->value('price');

        return (float) ($override ?? $test->default_price);
    }

    private function ensureOrderAccessible(Request $request, Order $order): void
    {
        $user = $request->user();
        if (in_array($user->role, ['lab_manager', 'lab_staff'], true) && (int) $order->lab_id !== (int) $user->lab_id) {
            abort(403, 'Unauthorized order access.');
        }
    }
}
