<?php

namespace App\Http\Controllers\Frontdesk;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class StaffController extends Controller
{
    public function index(Request $request): View
    {
        $manager = $request->user();

        $query = User::query()
            ->where('lab_id', $manager->lab_id)
            ->whereIn('role', ['lab_manager', 'lab_staff'])
            ->latest('id');

        if ($term = trim((string) $request->query('q'))) {
            $query->where(function ($q) use ($term): void {
                $q->where('name', 'like', "%{$term}%")
                    ->orWhere('email', 'like', "%{$term}%")
                    ->orWhere('phone', 'like', "%{$term}%")
                    ->orWhere('role', 'like', "%{$term}%");
            });
        }

        $staff = $query->paginate(15)->withQueryString();

        return view('frontdesk.staff.index', compact('staff'));
    }

    public function create(): View
    {
        return view('frontdesk.staff.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $manager = $request->user();
        $data = $this->validateStaff($request);

        User::query()->create([
            'name' => $data['name'],
            'email' => $data['email'],
            'phone' => $data['phone'] ?? null,
            'role' => $data['role'],
            'lab_id' => $manager->lab_id,
            'is_active' => (bool) ($data['is_active'] ?? true),
            'email_verified_at' => now(),
            'password' => Hash::make($data['password']),
        ]);

        return redirect()->route('frontdesk.staff.index')->with('success', 'Staff user created.');
    }

    public function edit(Request $request, User $user): View
    {
        $this->ensureSameLabStaff($request, $user);

        return view('frontdesk.staff.edit', compact('user'));
    }

    public function update(Request $request, User $user): RedirectResponse
    {
        $this->ensureSameLabStaff($request, $user);
        $data = $this->validateStaff($request, $user->id, false);

        $payload = [
            'name' => $data['name'],
            'email' => $data['email'],
            'phone' => $data['phone'] ?? null,
            'role' => $data['role'],
            'is_active' => (bool) ($data['is_active'] ?? false),
        ];

        if (! empty($data['password'])) {
            $payload['password'] = Hash::make($data['password']);
        }

        $user->update($payload);

        return redirect()->route('frontdesk.staff.index')->with('success', 'Staff user updated.');
    }

    public function switchRole(Request $request, User $user): RedirectResponse
    {
        $this->ensureSameLabStaff($request, $user);

        $data = $request->validate([
            'role' => ['required', Rule::in(['lab_manager', 'lab_staff'])],
        ]);

        $user->update([
            'role' => $data['role'],
        ]);

        return back()->with('success', "Role switched to {$data['role']} for {$user->name}.");
    }

    /**
     * @return array<string, mixed>
     */
    private function validateStaff(Request $request, ?int $userId = null, bool $requirePassword = true): array
    {
        $passwordRules = $requirePassword
            ? ['required', 'string', 'min:8', 'confirmed']
            : ['nullable', 'string', 'min:8', 'confirmed'];

        return $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users', 'email')->ignore($userId)],
            'phone' => ['nullable', 'string', 'max:30', Rule::unique('users', 'phone')->ignore($userId)],
            'role' => ['required', Rule::in(['lab_manager', 'lab_staff'])],
            'is_active' => ['nullable', 'boolean'],
            'password' => $passwordRules,
        ]);
    }

    private function ensureSameLabStaff(Request $request, User $user): void
    {
        $manager = $request->user();

        if ((int) $user->lab_id !== (int) $manager->lab_id || ! in_array($user->role, ['lab_manager', 'lab_staff'], true)) {
            abort(403, 'Unauthorized staff access.');
        }
    }
}
