<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Str;
use setasign\Fpdi\Fpdi;

class ResultWatermarkService
{
    public function storeWatermarked(UploadedFile $file, string $watermarkText): string
    {
        $extension = strtolower((string) $file->getClientOriginalExtension());
        $fileName = 'wm_'.Str::uuid()->toString().'.'.$extension;
        $relativePath = 'results/'.$fileName;
        $absolutePath = storage_path('app/public/'.$relativePath);
        $directory = dirname($absolutePath);

        if (! is_dir($directory)) {
            mkdir($directory, 0775, true);
        }

        if ($extension === 'pdf') {
            $this->watermarkPdf($file->getRealPath(), $absolutePath, $watermarkText);
            return $relativePath;
        }

        if (in_array($extension, ['jpg', 'jpeg', 'png', 'webp'], true)) {
            $this->watermarkImage($file->getRealPath(), $absolutePath, $extension, $watermarkText);
            return $relativePath;
        }

        $file->storeAs('results', $fileName, 'public');
        return $relativePath;
    }

    private function watermarkPdf(string $inputPath, string $outputPath, string $watermarkText): void
    {
        $pdf = new Fpdi();
        $pageCount = $pdf->setSourceFile($inputPath);

        for ($pageNo = 1; $pageNo <= $pageCount; $pageNo++) {
            $templateId = $pdf->importPage($pageNo);
            $size = $pdf->getTemplateSize($templateId);

            $orientation = $size['width'] > $size['height'] ? 'L' : 'P';
            $pdf->AddPage($orientation, [$size['width'], $size['height']]);
            $pdf->useTemplate($templateId);

            $pdf->SetTextColor(190, 190, 190);
            $pdf->SetFont('Helvetica', 'B', 28);
            $pdf->SetXY(10, $size['height'] / 2);
            $pdf->Cell($size['width'] - 20, 12, $watermarkText, 0, 0, 'C');
        }

        $pdf->Output('F', $outputPath);
    }

    private function watermarkImage(string $inputPath, string $outputPath, string $extension, string $watermarkText): void
    {
        $image = match ($extension) {
            'jpg', 'jpeg' => imagecreatefromjpeg($inputPath),
            'png' => imagecreatefrompng($inputPath),
            'webp' => imagecreatefromwebp($inputPath),
            default => null,
        };

        if (! $image) {
            copy($inputPath, $outputPath);
            return;
        }

        $width = imagesx($image);
        $height = imagesy($image);
        $color = imagecolorallocatealpha($image, 160, 160, 160, 70);
        $font = 5;
        $stepY = max((int) ($height / 4), 80);

        for ($y = (int) ($height * 0.2); $y < $height; $y += $stepY) {
            imagestring($image, $font, (int) ($width * 0.2), $y, $watermarkText, $color);
        }

        match ($extension) {
            'jpg', 'jpeg' => imagejpeg($image, $outputPath, 90),
            'png' => imagepng($image, $outputPath, 7),
            'webp' => imagewebp($image, $outputPath, 90),
            default => copy($inputPath, $outputPath),
        };

        imagedestroy($image);
    }
}
