<?php

namespace Database\Seeders;

use App\Models\Lab;
use App\Models\Patient;
use App\Models\TestCatalog;
use App\Models\TestCategory;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DemoDataSeeder extends Seeder
{
    /**
     * Demo login credentials for every user role present in the app.
     *
     * `lab_id` is assigned dynamically for non-super-admin users.
     *
     * @var array<int, array<string, mixed>>
     */
    private const LOGIN_ACCOUNTS = [
        [
            'name' => 'Super Admin',
            'email' => 'superadmin@lab.test',
            'phone' => '+2348000000001',
            'role' => 'super_admin',
            'uses_lab_id' => false,
            'password' => 'SuperAdmin@123',
        ],
        [
            'name' => 'Lab Manager',
            'email' => 'manager@lab.test',
            'phone' => '+2348000000002',
            'role' => 'lab_manager',
            'uses_lab_id' => true,
            'password' => 'LabManager@123',
        ],
        [
            'name' => 'Lab Staff',
            'email' => 'staff@lab.test',
            'phone' => '+2348000000003',
            'role' => 'lab_staff',
            'uses_lab_id' => true,
            'password' => 'LabStaff@123',
        ],
    ];

    /**
     * Demo patient contacts for OTP-only login.
     *
     * @var array<int, array<string, string>>
     */
    private const PATIENT_CONTACTS = [
        [
            'full_name' => 'John Patient',
            'email' => 'patient1@lab.test',
            'phone' => '+2348000000101',
            'gender' => 'male',
            'dob' => '1990-05-10',
        ],
        [
            'full_name' => 'Jane Patient',
            'email' => 'patient2@lab.test',
            'phone' => '+2348000000102',
            'gender' => 'female',
            'dob' => '1993-09-20',
        ],
    ];

    public function run(): void
    {
        $lab = Lab::query()->firstOrCreate(
            ['name' => 'VitalCare Diagnostics'],
            [
                'branch_name' => 'Central Branch',
                'email' => 'info@vitalcare.test',
                'phone' => '+2348000000000',
                'address' => '23 Health Avenue, Lagos',
                'is_active' => true,
            ]
        );

        $credentials = $this->seedUsers($lab->id);
        $patients = $this->seedPatients($lab->id);
        $this->seedTests();
        $this->outputCredentials($credentials);
        $this->outputPatientContacts($patients);
    }

    /**
     * @return array<int, array<string, string>>
     */
    private function seedUsers(int $labId): array
    {
        $credentials = [];

        foreach (self::LOGIN_ACCOUNTS as $data) {
            $plainPassword = $data['password'];

            User::query()->updateOrCreate(
                ['email' => $data['email']],
                [
                    'name' => $data['name'],
                    'phone' => $data['phone'],
                    'role' => $data['role'],
                    'lab_id' => $data['uses_lab_id'] ? $labId : null,
                    'is_active' => true,
                    'email_verified_at' => now(),
                    'password' => Hash::make($plainPassword),
                ]
            );

            $credentials[] = [
                'role' => $data['role'],
                'email' => $data['email'],
                'password' => $plainPassword,
            ];
        }

        return $credentials;
    }

    private function seedTests(): void
    {
        $categories = [
            'Hematology',
            'Chemistry',
            'Parasitology',
        ];

        foreach ($categories as $name) {
            TestCategory::query()->updateOrCreate(
                ['name' => $name],
                ['is_active' => true]
            );
        }

        $tests = [
            ['code' => 'CBC', 'name' => 'Complete Blood Count', 'category' => 'Hematology', 'default_price' => 5000],
            ['code' => 'MAL', 'name' => 'Malaria Test', 'category' => 'Parasitology', 'default_price' => 3000],
            ['code' => 'LFT', 'name' => 'Liver Function Test', 'category' => 'Chemistry', 'default_price' => 7500],
            ['code' => 'RFT', 'name' => 'Renal Function Test', 'category' => 'Chemistry', 'default_price' => 7000],
        ];

        foreach ($tests as $test) {
            $categoryId = TestCategory::query()
                ->where('name', $test['category'])
                ->value('id');

            TestCatalog::query()->updateOrCreate(
                ['code' => $test['code']],
                $test + ['category_id' => $categoryId, 'is_active' => true]
            );
        }
    }

    /**
     * @return array<int, array<string, string>>
     */
    private function seedPatients(int $labId): array
    {
        $contacts = [];

        foreach (self::PATIENT_CONTACTS as $data) {
            $patient = Patient::query()->updateOrCreate(
                ['email' => $data['email']],
                [
                    'lab_id' => $labId,
                    'full_name' => $data['full_name'],
                    'phone' => $data['phone'],
                    'gender' => $data['gender'],
                    'dob' => $data['dob'],
                ]
            );

            $contacts[] = [
                'name' => $patient->full_name,
                'email' => (string) $patient->email,
                'phone' => (string) $patient->phone,
            ];
        }

        return $contacts;
    }

    /**
     * @param array<int, array<string, string>> $credentials
     */
    private function outputCredentials(array $credentials): void
    {
        if (! $this->command) {
            return;
        }

        $this->command->newLine();
        $this->command->info('Seeded login details:');
        $this->command->table(
            ['Role', 'Email', 'Password'],
            array_map(fn (array $row): array => [
                $row['role'],
                $row['email'],
                $row['password'],
            ], $credentials)
        );
    }

    /**
     * @param array<int, array<string, string>> $contacts
     */
    private function outputPatientContacts(array $contacts): void
    {
        if (! $this->command || $contacts === []) {
            return;
        }

        $this->command->newLine();
        $this->command->info('Seeded patient OTP contacts:');
        $this->command->table(
            ['Patient', 'Email', 'Phone', 'Login Method'],
            array_map(static fn (array $row): array => [
                $row['name'],
                $row['email'],
                $row['phone'],
                'OTP only (email / SMS / WhatsApp)',
            ], $contacts)
        );
    }
}
